#ifndef __CAlphaColour__
#define __CAlphaColour__

#include "CColour.hpp"
using Exponent::GUI::Basics::CColour;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Basics
		{
			/**
			 * @class CAlphaColour CAlphaColour.hpp
			 * @brief Representation of an RGBA colour
			 *
			 * @date 20/09/2004
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 * @author Paul Chana
			 * @version 1.0.1 Added inplace getAsColourFunction
			 * @author Paul Chana
			 * @version 1.0.2 Made the equality operator const
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CAlphaColour.hpp,v 1.4 2007/02/08 21:07:54 paul Exp $
			 */
			class CAlphaColour : public CColour
			{
				/** @cond */
				EXPONENT_CLASS_DECLARATION;
				/** @endcond */

//	===========================================================================

			public:

//	===========================================================================

				const static CAlphaColour CALPHACOLOUR_BLACK;			/**< Black */
				const static CAlphaColour CALPHACOLOUR_WHITE;			/**< White */
				const static CAlphaColour CALPHACOLOUR_RED;				/**< Red */
				const static CAlphaColour CALPHACOLOUR_GREEN;			/**< Green */
				const static CAlphaColour CALPHACOLOUR_BLUE;			/**< Blue */
				const static CAlphaColour CALPHACOLOUR_YELLOW;			/**< Yellow */
				const static CAlphaColour CALPHACOLOUR_ORANGE;			/**< Orange */
				const static CAlphaColour CALPHACOLOUR_TURQUOISE;		/**< Turqoise */
				const static CAlphaColour CALPHACOLOUR_PURPLE;			/**< Purple */
				const static CAlphaColour CALPHACOLOUR_PINK;			/**< Pink */
				const static CAlphaColour CALPHACOLOUR_LIGHT_GREY;		/**< Light grey */
				const static CAlphaColour CALPHACOLOUR_DARK_GREY;		/**< Dark grey */

//	===========================================================================

				/**
				 * Construction
				 * @param red The red component
				 * @param green The green component
				 * @param blue The blue component
				 * @param alpha The alpha component
				 */
				CAlphaColour(const unsigned long red, const unsigned long green, const unsigned long blue, const unsigned long alpha = 255);

				/**
				 * Construction
				 */
				CAlphaColour();

				/**
				 * Copy construction
				 * @param colour The colour to copy
				 */
				CAlphaColour(const CAlphaColour &colour);

				/**
				 * Destruction
				 */
				virtual ~CAlphaColour();

//	===========================================================================

				/**
				 * Assignment operator
				 * @param colour The colour to copy
				 */
				CAlphaColour &operator = (const CAlphaColour &colour);

				/**
				 * Assignment operator
				 * @param colour The colour to copy with alpha 255
				 */
				CAlphaColour &operator = (const CColour &colour);

				/**
				 * Equality operator
				 * @param colour The colour to compare to
				 * @retval bool True if colours are the same, false otherwise
				 */
				bool operator == (const CAlphaColour &colour) const;

//	===========================================================================

				/**
				 * Set the alpha value
				 * @param alpha The alpha component
				 */
				void setAlpha(const unsigned long alpha) { m_alpha = alpha; }

				/**
				 * Get the alpha value
				 * @retval unsigned long The alpha value
				 */
				unsigned long getAlpha() const { return m_alpha; }

//	===========================================================================

				/**
				 * Darken the colour
				 */
				void darkenColour();

				/**
				 * Lighten the colour
				 */
				void lightenColour();

//  ============================================================================

				/**
				 * Set all the values
				 * @param red The red component
				 * @param green The green component
				 * @param blue The blue component
				 * @param alpha The alpha component
				 */
				virtual void setColour(const unsigned long red, const unsigned long green, const unsigned long blue, const unsigned long alpha);

				/**
				 * Get as a colour
				 * @retval CColour* The colour, you must dispose
				 */
				CColour *getAsColour() const;

				/**
				 * Get as a colour
				 * @param colour On return this is filled with the colour
				 */
				void getAsColour(CColour &colour) const;

//	===========================================================================

				/**
				 * Alpha blend two colours
				 * @param destination The destination colour, on return this is filled with the alpha blended colour
				 * @param background The background colour
				 * @param source The top level colour
				 */
				static FORCEINLINE void alphaBlend(CColour &destination, const CAlphaColour &background, const CAlphaColour &source)
				{
					destination.setRed(CAlphaColour::alphaComposite(source.getRed(), background.getRed(), source.getAlpha()));
					destination.setGreen(CAlphaColour::alphaComposite(source.getGreen(), background.getGreen(), source.getAlpha()));
					destination.setBlue(CAlphaColour::alphaComposite(source.getBlue(), background.getBlue(), source.getAlpha()));
				}

				/**
				 * Alpha blend two colours with a fixed alpha
				 * @param destination The destination colour, on return this is filled with the alpha blended colour
				 * @param background The background colour
				 * @param source The top level colour
				 * @param alpha The alpha level to process with
				 */
				static FORCEINLINE void alphaBlend(CColour &destination, const CColour &background, const CColour &source, const unsigned long alpha)
				{
					destination.setRed(CAlphaColour::alphaComposite(source.getRed(), background.getRed(), alpha));
					destination.setGreen(CAlphaColour::alphaComposite(source.getGreen(), background.getGreen(), alpha));
					destination.setBlue(CAlphaColour::alphaComposite(source.getBlue(), background.getBlue(), alpha));
				}

//	===========================================================================

				/**
				 * Get a description of the object
				 * @param string On return is filled with the description
				 * @param size The size of the stirng
				 */
				virtual void getObjectDescription(char *string, const long size) const;

//	===========================================================================

			protected:

//	===========================================================================

				/**
				 * Alpha composite a single value
				 * @param foreground The front colour
				 * @param background The background colour
				 * @param alpha The alpha value
				 */
				static FORCEINLINE unsigned long alphaComposite(const unsigned long foreground, const unsigned long background, const unsigned long alpha)
				{
					unsigned long temp = ((unsigned long)(foreground)*(unsigned long)(alpha) + (unsigned long)(background)*(unsigned long)(255 - (unsigned long)(alpha)) + (unsigned long)128); 
					return (unsigned long)((temp + (temp >> 8)) >> 8);
				}

//	===========================================================================

				unsigned long m_alpha;		/**< Alpha value (0 - 255) */

			};
		}
	}
}
#endif	// end of CAlphaColour.hpp